<?php

declare(strict_types=1);

namespace Erlage\Photogram\Tools\Fso;

use Throwable;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Tools\Fso\Disk\DiskS3;
use Erlage\Photogram\Tools\Fso\Disk\DiskLocal;
use Erlage\Photogram\Tools\Fso\Disk\DiskWasabi;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Tools\Fso\Disk\InterfaceDisk;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Data\Models\Setting\SettingEnum;

class Storage
{
    /**
     * @var string
     */
    private $objectId;

    /**
     * @var InterfaceDisk
     */
    private $disk;

    /**
     * List of resolved storage instances.
     * 
     * @var array
     */
    private static $disks = array();

    /*
    |--------------------------------------------------------------------------
    | constructor
    |--------------------------------------------------------------------------
    */

    public function __construct(string $diskName)
    {
        switch ($diskName)
        {
            case SettingEnum::STORAGE_DISK_LOCAL: $this -> disk = new DiskLocal();

            break;

            case SettingEnum::STORAGE_DISK_AWS: $this -> disk = new DiskS3();

            break;

            case SettingEnum::STORAGE_DISK_WASABI: $this -> disk = new DiskWasabi();

            break;

            default:
            throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | instance resolver
    |--------------------------------------------------------------------------
    */

    public static function disk(string $name): self
    {
        if ( ! isset(self::$disks[$name]))
        {
            self::$disks[$name] = new self($name);
        }

        return self::$disks[$name];
    }

    /*
    |--------------------------------------------------------------------------
    | setters
    |--------------------------------------------------------------------------
    */

    public function setObjectId(string $objectId): self
    {
        $this -> objectId = $objectId;

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | interface
    |--------------------------------------------------------------------------
    */

    public function write(string $contents): void
    {
        $this -> disk -> write($this -> objectId, $contents);
    }

    public function read(): string
    {
        return $this -> disk -> read($this -> objectId);
    }

    public function delete(): void
    {
        try
        {
            $this -> disk -> delete($this -> objectId);
        }
        catch (Throwable $exception)
        {
            SystemLogger::internalException($exception);
        }
    }

    /**
     * Delete objects referenced by DTO (only if they are not placeholder) 
     */
    public function deleteDTO(DisplayItemDTO $displayItemDTO): void
    {
        if (DisplayItemDTO::FILESPACE_PLACEHOLDER != $displayItemDTO -> getFilespace())
        {
            $this -> setObjectId($displayItemDTO -> getOriginalObjectId());
            $this -> delete();

            $this -> setObjectId($displayItemDTO -> getCompressedObjectId());
            $this -> delete();
        }
    }

    public function getPublicUrl(): string
    {
        return $this -> disk -> publicUrl($this -> objectId);
    }
}
